/*
 *  HP 9000 Series 800 Linker, Copyright Hewlett-Packard Co. 1985-1999  
***************************************************************************
**
** File:         debug_opt.c
**
** Description:  Debug line table support routines. 
**
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
***************************************************************************
*/

/*
**************************************************************************
**	INCLUDE FILES	
**************************************************************************
*/
#include <stdlib.h>
#include <stdio.h>

#include "filehdr.h"
#include "spacehdr.h"
#include "scnhdr.h"
#include "reloc.h"
#include "syms.h"

#include "std.h"
#include "driver.h"
#include "ldlimits.h"
#include "linker.h"
#include "spaces.h"
#include "subspaces.h"
#include "output.h"
#include "symbols.h"
#include "fixups.h"
#include "util.h"
#include "debug_opt.h"

/* doom support */
#include "ld_linkmap.h"

/*
**************************************************************************
**	DEFINE VALUES	
**************************************************************************
*/
#define DOC_TABLE_SIZE		1024

#define FLUSH_SIZE		DOC_TABLE_SIZE + 1
#define VERSION_SIZE		1
#define ESCAPE_ENTRY_SIZE	1
#define ESCAPE_DATA_SIZE	1
#define ENTRY_SIZE		1
#define PAD_SIZE		1

#define DST_LN_END		0xff
#define DST_LN_PAD		0xf0
#define DST_LN_DPC1_DLN1	0xf2
#define DST_LN_DPC2_DLN2	0xf3
#define DST_LN_PC4_LN4		0xf4
#define DST_LN_DPC0_DLN1	0xf5

/*
**************************************************************************
**	EXTERNAL	
**************************************************************************
*/
int	doc_state;		/* state of debug line tables mechanism	*/ 
size_t	doc_table_size;		/* size of the all the line tables	*/
int	doc_lines_subsp;	/* subspace index of $LINES$ subspace	*/

/*
**************************************************************************
**	LOCAL	
**************************************************************************
*/

struct line_table_info {
    int patch_symbol_index;	/* symbol index to be patched */
    int	patch_symbol_offset;	/* symbol relative offset     */
    int	line_table_offset;	/* subspace relative offset   */
};

/*
** number of line tables encountered.
*/
static unsigned int		num_line_tables;

/*
** When building the debug line tables cur_line_num and cur_line_addr
** represent the current line number and absolute address values while 
** generating the tables.  Note! cur_line_addr is a word address not
** a byte address.
*/
static unsigned int		cur_line_num;
static unsigned int		cur_line_addr;

/*
** cur_index represents the current index into the doc_info table.  Each
** debug line table will have its own doc_info record.
*/
static unsigned int		cur_index;

/*
** cur_version represents the version of the debug line table being 
** processed.
*/
static unsigned char		cur_version;

/*
** escape_data_count contains the number of R_STATEMENT fixups to be
** processed as escape data instead of statement numbers.
*/
static unsigned int		escape_data_count;

/*
** doc_subsp_offset holds the current offset within the $LINES$ subspace.
*/
static int			doc_subsp_offset;


/*
** output_offset represents the offset within the doc_line_table_output
** buffer.
*/
static int			output_offset;

/*
** output buffer for debug line tables.  debug line tables entries are
** buffered before they are written to the output file.
*/
static unsigned char		*doc_line_table_output;

/*
** doc_info points to the debug line table information records which
** contain locations of all the debug line tables within the $LINES$
** subspace.
*/
static struct line_table_info 	*doc_info;

/*
** doc_table_file_loc holds the current output file location for 
** the output of debug line tables.  It is initialialized to the
** location of the start of the $LINES$ subspace in the output file.
*/
static unsigned int 		doc_table_file_loc;

/* 
** need_first_statement is set to TRUE in doc_count_new and doc_gen_new
** when an R_LINETAB fixup is processed.  It is set to FALSE in 
** doc_count_entry and doc_gen_entry when the first R_STATEMENT is processed
** that is not part of a R_LINETAB_ESC.
*/

static Boolean			need_first_statement;

/*
**************************************************************************
**
** doc_init_pass1()
**
** This routine determines if debug line tables can be expected in the
** input object files.  This is done by checking to see if the 
** $DEBUG$ space has been seen.  Also, we check to see if debug
** information is being stripped.  If we have the $DEBUG$ space and
** we haven't seen an option to strip the debug info then we initialize 
** DOC structures and generate a $LINES$ subspace in preparation 
** for the first pass over the fixups to determine the size of the 
** line tables. 
**
**************************************************************************
*/
void doc_init_pass1(void)
{
    int	space_index;
    struct subspace_dictionary_record *subsp;

    /*
    ** initialize all externally visible values including the
    ** state of the line table mechanism.
    */
    doc_state = _DOC_INIT;
    doc_table_size = 0;
    doc_lines_subsp = BAD_SUBSP;


    /*
    ** disable building line tables if stripping debug or performing
    ** a relocatable link.  Note! no warning is issued at this point.
    ** For any conditions which require a warning see the following
    ** checks.
    */
    if (lm_should_disable_linetables() || strip_debug || relocatable) {
	doc_state = _DOC_DISABLED;
	return;
    } 

    /*
    ** since linkmap will invoke pxdb only if the debug info
    ** is copied to the output SOM file, the following will be
    ** a faster version of the determining whether object file
    ** contain any $DEBUG$ spaces.
    */
    if (!lm_should_invoke_pxdb()) {
       doc_state = _DOC_DISABLED;	
       return;
    }
	
#if 0
    /*
    ** determine if the $DEBUG$ space is present.  If not then disable
    ** line table generation.
    */
    for (space_index = 0; space_index < cur_space_total; space_index++) {
        if (!space_array[space_index].is_loadable &&
            (strcmp(space_array[space_index].name.n_name, "$DEBUG$") 
            == 0)) {
            break;
        }
    }

    if (space_index >= cur_space_total) {
	doc_state = _DOC_DISABLED;
	return;
    }
#endif 
    
    /*
    ** build the $LINES$ subspace in the $DEBUG$ space. This
    ** subspace is word aligned just like other $DEBUG$ subspaces.
    ** The quadrant, and access info bits do not matter since this
    ** subspace is not loadable.
    */
    doc_lines_subsp = build_subspace("$LINES$",
				     "$DEBUG$",
				     DEBUG_SORTKEY,
				     0,           
				     0, 
				     SORT_FIRST, 
				     4, 
				     0, 
				     0);

    /*
    ** check to make sure the subspace was successfully created.  If not
    ** issue a warning and change doc_state to _DOC_ERROR.
    */
    if (doc_lines_subsp == BAD_SUBSP) {
        doc_state = _DOC_ERROR;	
	return;
    }

    /*
    ** Correct some of the subspace information.  The $LINES$ 
    ** subspace is not a loadable subspace.
    */
    subsp = & Subsp_Dict(doc_lines_subsp);
    subsp->is_loadable = FALSE;

    /*
    ** Initialize anything needed for the first pass over the 
    ** fixups.
    */
    num_line_tables = 0;
    cur_index = 0;
    cur_line_num = 0;
    cur_line_addr = 0;
    cur_version = 0;
    escape_data_count = 0;

}


/*
**************************************************************************
**
** doc_post_pass1()
**
** Routine to set $LINES$ subspace length.  Also checks to see if first 
** pass over the fixups caused doc_state to end in the proper state.  
** We do not know if DOC line tables are to be generated until after the 
** first pass of the fixups.  The reason for this is that we determine if
** debug line tables are to be generated by the fixup entries.  So we must
** complete at least one pass to determine if they are to be generated.
** If any incompatible options or features were specified then issue a 
** diagnostic and disable DOC line tables.
**
** This routine must be called before virtual addresses or file locations
** are set in assign_virtual_addresse() or assign_file_locations().
**
**************************************************************************
*/
void doc_post_pass1(void)
{
    struct subspace_dictionary_record *subsp;

    /*
    ** If an error occured in the first pass doc_state will most likely
    ** be set to _DOC_ERROR.  Verify that the first pass over the fixups 
    ** culminated in the line table doc_state to be in the proper state 
    ** of _DOC_SEEN.  If not do not generate line tables.
    **
    ** If doc_init (we can assume zero length $LINES$ subspace but we'll
    ** check anyway), set bit in misc record so we don't output the
    ** subspace.  We only want to NOT output $LINES$ if we are doing a NON
    ** DOC compile.  Even though we disable DOC for ADDIL and aux unwind,
    ** we will still output a zero-length $LINES$ subspace (i.e. if we 
    ** did a DOC compile and did ADDIL we would still output $LINES$).
    */

    if (doc_state == _DOC_INIT &&
        doc_table_size == 0) {
        SUBSP_DELETE (doc_lines_subsp);
        doc_state = _DOC_DISABLED;
        return;
    }

    if (doc_state != _DOC_SEEN) {
        doc_state = _DOC_DISABLED;
	warning(1577,0);
        SUBSP_DELETE (doc_lines_subsp);
	return;
    } 

    /*
    ** if doing ADDIL elimination issue a warning and disable DOC
    ** line tables. DOC not supported with -A option or FRU relink. 
    */

    if (eliminating_addils || base_file_name != NULL) {
        doc_state = _DOC_DISABLED;
	warning(1578,0);
        return;
    }

    /*
    ** If generating aux unwind entries disable DOC line tables.
    */
    if (generate_aux_unwind) {
	doc_state = _DOC_DISABLED;
	warning(1579, 0);
	return;
    }

    /*
    ** Set the size of the $LINES$ subspace to reflect the size of the
    ** line tables to be generated in the final pass over the fixups.
    ** If we're here, $LINES$ is "safe" to output--either zero-length and
    ** we're doing ADDIL or generating aux unwind *or* we have DOC line
    ** tables.
    */
    if (doc_lines_subsp != BAD_SUBSP) {
        subsp = & Subsp_Dict(doc_lines_subsp);
        subsp->initialization_length = subsp->subspace_length = doc_table_size;
    }
}


/*
**************************************************************************
**
** doc_init_pass2()
**
** Routine to initialize DOC line table structures prior the final pass
** over the fixups.  In the final pass over the fixups the line tables
** are actually generated and written to the executable file.
** Allocate any storage necessary for the final pass such as output 
** buffers, etc.  
**
**************************************************************************
*/
void doc_init_pass2(void)
{
    doc_subsp_offset = 0;
    output_offset = 0;
    cur_line_num = 0;
    cur_line_addr = 0;
    cur_index = 0;
    cur_version = 0;
    escape_data_count = 0;
    doc_table_file_loc = Subsp_Misc(doc_lines_subsp).exec_file_offset;

    /*
    ** Allocate the info buffer which keeps track of the the line
    ** table offsets and the location where the table offsets are
    ** to be written. 
    ** We allocate on extra entry so that doc_gen_new can record a line
    ** table at the end which is not counted because it doesn't have
    ** any contents.
    */
    doc_info = (struct line_table_info *) 
        emalloc((num_line_tables + 1) * sizeof(struct line_table_info));

    /*
    ** Allocate the buffer used to output the debug line table entries.
    */
    doc_line_table_output = (unsigned char *) emalloc(DOC_TABLE_SIZE);
}


/*
**************************************************************************
**
** doc_output()
**
** Utility routine to write out actual line tables.  Output is buffered, so
** checks are made to determine if current output request will overflow
** the output buffer.  If so the buffer is written to the executable file
** and reinitialized for more table entries.  
**
** Note! to avoid any funny behaviour, always use the maximum size 
** possible for a given entry.
**
**************************************************************************
*/
static void doc_output(int size)
{
    /*
    ** If the output buffer will overflow then write it out to the 
    ** output file and reset the buffer.
    */
    if (output_offset != 0 && (output_offset + size) >= DOC_TABLE_SIZE) {
	fdump(out_som_fd, 
	      doc_table_file_loc,
	      doc_line_table_output,
	      output_offset,
	      sizeof(unsigned char));
        doc_table_file_loc += output_offset;
	output_offset = 0;
    }
}

/*
**************************************************************************
**
** doc_count_escape()
**
** Called when a R_LINETAB_ESC fixup is encountered.  This routine increments 
** the line table size by the size of the escape and sets the number of
** R_STATEMENT fixups which will be processed as escape data entries.
** Refer to the doc_count_new() for the _DOC_ESC state switch for details.
**
**************************************************************************
*/
void doc_count_escape(unsigned int num_data)
{
    doc_table_size += ESCAPE_ENTRY_SIZE;

    /*
    ** Do not change state if there are no escape data entries.
    */
    if ((escape_data_count = num_data) != 0) {
	/* 
	** If the state is _DOC_NEW, increment num_line_tables.
	*/

	if (doc_state == _DOC_NEW) {
	    num_line_tables++;
	}
        doc_state = _DOC_ESC; 
    }
}


/*
**************************************************************************
**
** doc_gen_escape()
**
** Called when a R_LINETAB_ESC fixup is encountered in the final pass over 
** the fixups.  Output the escape which is requested.  Also, set the number
** of R_STATEMENT fixups which will be processed as escape data entries.
** Change the doc_state to _DOC_ESC.
** 
**************************************************************************
*/
void doc_gen_escape(unsigned char escape, unsigned int num_data)
{
    doc_output(ESCAPE_ENTRY_SIZE);
    doc_line_table_output[output_offset++] = escape;
    doc_subsp_offset++;

    /*
    ** Do not change state if there are no escape data entries.
    */
    if ((escape_data_count = num_data) != 0) {
	/*
	** If the state is _DOC_NEW, increment cur_index.
	*/
	if (doc_state == _DOC_NEW) {
	    cur_index++;
	}

        doc_state = _DOC_ESC;
    }
}


/*
**************************************************************************
**
** doc_count_end()
**
** Count the line table end escape entry and any padding to be generated after
** the line table.  Check the state of the line table.  If were currently
** in the middle of an escape sequence an error has occured so disable the 
** line table mechanism.
**
**************************************************************************
*/
void doc_count_end(void)
{
    switch (doc_state) {
        case _DOC_BUILD:
	    doc_table_size += ESCAPE_ENTRY_SIZE;

    	    /*
    	    ** Start all new line tables on word boundaries. Add padding if
    	    ** necessary.
    	    */
	    while (doc_table_size % WORDSIZE)
	        doc_table_size++;

            doc_state = _DOC_SEEN;
	    break;
        case _DOC_ESC:
	    doc_state = _DOC_ERROR;
	    return;
	    break;
        default:
	    /* do nothing */
	    break;
    }
}


/*
**************************************************************************
**
** doc_gen_end()
**
** If were currently building a line table then generate a dst_ln_end 
** escape entry to mark the end of the current line table.  Also, pad
** out to a word boundary such that the next line table if there is one
** will be aligned.  If were currently processing an escape sequence 
** an error has occured.  This will cause the line table mechanism to 
** become disabled.
**
**************************************************************************
*/
void doc_gen_end(void)
{
    switch (doc_state) {
        case _DOC_BUILD:
	    /*
	    ** Make sure there is enough room in the buffer to handle the
	    ** end escape entry including any padding.
	    */
	    doc_output(ESCAPE_ENTRY_SIZE + (WORDSIZE * PAD_SIZE));

	    doc_line_table_output[output_offset++] = DST_LN_END;
	    doc_subsp_offset++;

	    while (doc_subsp_offset % WORDSIZE) {
	        doc_line_table_output[output_offset++] = DST_LN_PAD;
	        doc_subsp_offset++;
            }

    	    doc_state = _DOC_SEEN;
	    break;
        case _DOC_ESC:
	    doc_state = _DOC_DISABLED; 
	    external_error(3011,0);
	    return;
	    break;
        default:
	    /* do nothing */
	    break;
    }
}


/*
**************************************************************************
**
** doc_count_new()
**
** When a R_LINETAB fixup is encountered it specifies that a new line table
** should be generated.  If in the first pass over the fixups we call
** doc_count_end() to verify that we are in the proper state and then
** we change the doc_state to _DOC_NEW.  doc_count_entry() wil actually
** determine if a new line table will be generated because we need to
** see an R_STATEMENT fixup to verify that the new line table has any
** entries.  Therefore delay counting the new line table escape until
** doc_count_entry().
**
**************************************************************************
*/
void doc_count_new(void)
{
    doc_count_end();
    doc_state = _DOC_NEW;
    need_first_statement = TRUE;  

}


/*
**************************************************************************
**
** doc_gen_new()
**
** Routine is called when a R_LINETAB fixup is encountered in the final
** pass over the fixups.  Set the version which was an argument to the
** fixup to the cur_version.  Also, set the doc_info structure for this
** line table.  This information will be used later when patching
** the line table offsets.  See doc_finish() for details on patching
** line table offsets.   Call doc_gen_end() to verify that doc_state is
** in the proper state and generate an end escape for the previous line
** table if necessary.  Lastly, set doc_state to _DOC_NEW.
**
**************************************************************************
*/
void doc_gen_new(unsigned char version, 
		 unsigned int symbol_index, 
		 int symbol_offset)
{
    cur_version = version;

    /*
    ** Set up the line table info for later use when patching
    ** line table offsets.  
    **
    ** Note! cur_index is incremented when
    ** the first entry is encountered.  No line table is 
    ** generated until the first R_STATEMENT fixup is seen.
    */
    doc_info[cur_index].patch_symbol_index = symbol_index;
    doc_info[cur_index].patch_symbol_offset = symbol_offset;
    doc_info[cur_index].line_table_offset = doc_subsp_offset;

    doc_gen_end();
    doc_state = _DOC_NEW;
    need_first_statement = TRUE; 

}

/*
**************************************************************************
**
** doc_count_entry()
**
** When a R_STATEMENT fixup is encountered in the first pass this routine
** is called to count the size of the line table entry.  Depending on
** the doc_state, we could be either counting a data escape, regular
** line table entry, or a new line table escape entry.  If this is
** a data escape line_num will represent one byte of data to be counted.
**
**************************************************************************
*/
void doc_count_entry(unsigned int line_addr, unsigned int line_num)
{
    int			line_num_delta;
    int			line_addr_delta;
    unsigned int	line_num_overflow;
    unsigned int	line_addr_overflow;

    switch (doc_state) {

	case _DOC_NEW: 

            /*
            ** check the state of the line table.  No reason to 
	    ** build a line table if there are no statements.  So we 
	    ** build the initial line table information when the first 
	    ** R_STATEMENT fixup is encountered.
	    **
	    ** When building a new line table the table should be word
	    ** aligned.  Note that the first table in the subspace will
	    ** be aligned since the subspace is word aligned.  All other
	    ** tables may need to be aligned.  Padding is taken into
	    ** consideration in doc_count_end().  doc_count_end() also
	    ** takes padding at the end of the subspace into account.
            */

    	    doc_state = _DOC_BUILD;
	    num_line_tables++;
	    /* Fall into _DOC_BUILD case, need_first_statement will be TRUE */

        case _DOC_BUILD: 
	    if (need_first_statement) {
		/** Handle special-case processing for the first real R_STATEMENT
		*/
	     
		need_first_statement = FALSE;
		cur_line_num = line_num;
		cur_line_addr = line_addr;


		/*
		** Count version entry size and 4 byte absolute code address 
		** followed by a 1-to-4 byte absolute line number.  
		** The line number size will be the minimum necessary to hold 
		** the value.
		*/
		doc_table_size += VERSION_SIZE; 
		doc_table_size += WORDSIZE;
		doc_table_size += (line_num > 0xffffff) ? 4 :
				  (line_num > 0xffff) ? 3 :
				  (line_num > 0xff) ? 2 : 1;
	    } else { /* ! need_first_statement */

		/*
		** Count the line table entry size associated with this 
		** R_STATEMENT fixup.
		*/

		/*
		** Determine the code and line delta values. Note! that the
		** code delta's will be a word address, where absolute addresses
		** will be byte addresses.
		*/
		line_num_delta = line_num - cur_line_num;
		if ((line_addr_delta = (line_addr - cur_line_addr)) < 0) {
		    doc_state = _DOC_ERROR;
		    return;
		}

		/*
		** Note! the current line address is a word address, not a 
		** byte address.
		*/
		cur_line_addr = line_addr; 
		cur_line_num = line_num; 

		/*
		** determine the size of the entry for the code and 
		** line delta's
		**
		** The size in bytes of the entry is determined in the 
		** following table.  If either code delta or line delta 
		** goes out of range the next largest entry is used for 
		** both line and code delta.
		**
		** code delta	line delta
		** range		range			size
		** +----------+---------------------------+--------+
		** |0..11     | -8..7		      |	1      |
		** +----------+---------------------------+--------+
		** |12..255   | -128..-9, 8..127          |	3      |
		** +----------+---------------------------+--------+
		** |256..65535| -32768..-129, 128..32767  |	5      |
		** +----------+---------------------------+--------+
		** |> 65535   | < -32768, > 32767         |	9      |	
		** +----------+---------------------------+--------+
		**
		*/
		line_num_overflow = line_addr_overflow = 0;

		if (line_addr_delta > 11) {
		    line_addr_overflow = (line_addr_delta > 0x0000ffff) ? 4 : 
					 (line_addr_delta > 0x000000ff) ? 2 : 1;
		}
	    
		if (line_num_delta > 7) {
		    line_num_overflow = (line_num_delta > 0x00007fff) ? 4 :
					(line_num_delta > 0x0000007f) ? 2 : 1;

		} else if (line_num_delta < -8) {
		    line_num_overflow = (line_num_delta < -0x00008000) ? 4 :
					(line_num_delta < -0x00000080) ? 2 : 1;
		}

		if (line_addr_delta == 0 && line_num_overflow == 1) {
		    doc_table_size += ESCAPE_ENTRY_SIZE + ENTRY_SIZE;
		} else {
		    doc_table_size += 
			(((line_addr_overflow > line_num_overflow) ?
			  line_addr_overflow 
			: line_num_overflow) * 2) + ENTRY_SIZE; 
		}

	    } /* end else ! need_first_statement */
	    break;

        case _DOC_ESC:
	    /*
	    ** Currently processing a R_LINETAB_ESC fixup. Instead of 
	    ** generating the code and line delta, handle the value as
	    ** absolute data requiring only one byte.
	    **
            ** Check to make sure escape entries have no following data.
	    */
	    if (escape_data_count > 0) {
	        escape_data_count--;
	        doc_table_size += ESCAPE_DATA_SIZE;
	    }

	    if (escape_data_count == 0)
		doc_state = _DOC_BUILD;
	    break;

        default:
	    doc_state = _DOC_ERROR;
	    return;
	    break;

    } /* end switch */
} /* end doc_count_entry */

/*
**************************************************************************
** 
** doc_gen_entry()
**
** When a R_STATEMENT fixup is encountered in the last pass this routine
** is called to generate the line table entry.  Depending on
** the doc_state, we could be either generating a data escape, regular
** line table entry, or a new line table escape entry.  If this is
** a data escape line_num will represent one byte of raw data to entered
** directly into the line table.
**
**************************************************************************
*/
void doc_gen_entry(unsigned int line_addr, unsigned int line_num)
{
    union {
	struct {
	    unsigned char	addr_delta:4;
	    char 		line_delta:4;
	} nibble;
	unsigned char byte;
    } regular_entry;

    int			entry_size;
    int 		line_num_delta;
    int 		line_addr_delta;
    unsigned int	line_num_overflow;
    unsigned int	line_addr_overflow;

    switch (doc_state) {

	case _DOC_NEW: 
	    /*
	    ** When building a new line table the table should be word
	    ** aligned.  Note! that the first table in the subspace will
	    ** be aligned since the subspace is word aligned.  All other
	    ** tables may need to be aligned.  Padding is added  
	    ** in doc_gen_end().  doc_gen_end() also generates padding 
	    ** at the end of the subspace if necessary.
            */

	    cur_index++;
            doc_state = _DOC_BUILD;
	    /* Fall into _DOC_BUILD case, need_first_statement will be TRUE */

	case _DOC_BUILD:
	    if (need_first_statement) {
	        /** Handle special-case processing for the first real R_STATEMENT
	        */
	     
		need_first_statement = FALSE;
		cur_line_num = line_num;
		cur_line_addr = line_addr;

		/*
		** flush the output buffer if it can't handle a new table
		** entry.  Assume 4 byte absolute line number when trying
		** to determine whether or not to flush.
		*/
		doc_output(VERSION_SIZE + WORDSIZE + WORDSIZE);

		/*
		** Start the new line table with a one byte version entry.
		*/
		doc_line_table_output[output_offset++] = cur_version; 
		doc_subsp_offset++;

		/*
		** Enter the absolute 4 byte code address. Note! we will
		** convert the word address to a byte address before writing.
		*/
		line_addr = line_addr * WORDSIZE;
		doc_line_table_output[output_offset++] = (line_addr >> 24);
		doc_line_table_output[output_offset++] = (line_addr >> 16);
		doc_line_table_output[output_offset++] = (line_addr >> 8);
		doc_line_table_output[output_offset++] = line_addr;
		doc_subsp_offset += 4;

		/*
		** Enter 1 to 4 byte absolute line number.
		*/
		if (line_num > 0xffffff) {
		    doc_line_table_output[output_offset++] = (line_num >> 24);
		    doc_subsp_offset++;
		}

		if (line_num > 0xffff) {
		    doc_line_table_output[output_offset++] = (line_num >> 16);
		    doc_subsp_offset++;
		}

		if (line_num > 0xff) {
		    doc_line_table_output[output_offset++] = (line_num >> 8);
		    doc_subsp_offset++;
		}

		doc_line_table_output[output_offset++] = line_num;
		doc_subsp_offset++;
	    } else { /* ! need_first_statement */

		/*
		** Determine the code and line delta values.
		*/
		line_num_delta = line_num - cur_line_num;
		if ((line_addr_delta = (line_addr - cur_line_addr)) < 0) {
		    doc_state = _DOC_DISABLED;
		    external_error(3011,0);
		    return;
		}

		entry_size = line_num_overflow = line_addr_overflow = 0;

		if (line_addr_delta > 11) {
		    line_addr_overflow = (line_addr_delta > 0x0000ffff) ? 4 :
					 (line_addr_delta > 0x000000ff) ? 2 : 1;
		}

		if (line_num_delta > 7) {
		    line_num_overflow = (line_num_delta > 0x00007fff) ? 4 :
					(line_num_delta > 0x0000007f) ? 2 : 1;

		} else if (line_num_delta < -8) {
		    line_num_overflow = (line_num_delta < -0x00008000) ? 4 :
					(line_num_delta < -0x00000080) ? 2 : 1;
		}

		if (line_addr_delta == 0 && line_num_overflow == 1) {
		    entry_size = ESCAPE_ENTRY_SIZE + ENTRY_SIZE;
		} else {
		    entry_size = (((line_addr_overflow > line_num_overflow) ? 
			line_addr_overflow : line_num_overflow) * 2) + 
			ESCAPE_ENTRY_SIZE;
		}

		cur_line_addr = line_addr;
		cur_line_num = line_num;

		doc_output(entry_size);

		switch (entry_size) {

		    case 9:
			doc_line_table_output[output_offset++] = DST_LN_PC4_LN4;
			/*
			** follow with an absolute address followed by a
			** absolute 4 byte line number. Note! we will convert 
			** the line number to a byte address instead of a word 
			** address
			*/
			line_addr = line_addr * WORDSIZE;
			doc_line_table_output[output_offset++] = 
				(line_addr >> 24);
			doc_line_table_output[output_offset++] = 
				(line_addr >> 16);
			doc_line_table_output[output_offset++] = 
				(line_addr >> 8);
			doc_line_table_output[output_offset++] = line_addr;
			doc_line_table_output[output_offset++] = 
				(line_num >> 24);
			doc_line_table_output[output_offset++] = 
				(line_num >> 16);
			doc_line_table_output[output_offset++] = 
				(line_num >> 8);
			doc_line_table_output[output_offset++] = line_num;
			doc_subsp_offset += 9;
			break;
			
		    case 5:
			doc_line_table_output[output_offset++] = 
			    DST_LN_DPC2_DLN2;
			doc_line_table_output[output_offset++] = 
			    (line_addr_delta >> 8);
			doc_line_table_output[output_offset++] = 
			    line_addr_delta;
			doc_line_table_output[output_offset++] = 
			    (line_num_delta >> 8);
			doc_line_table_output[output_offset++] = 
			    line_num_delta;
			doc_subsp_offset += 5;
			break;

		    case 3:
			doc_line_table_output[output_offset++] 
				= DST_LN_DPC1_DLN1;
			doc_line_table_output[output_offset++] 
				= line_addr_delta;
			doc_line_table_output[output_offset++] = line_num_delta;
			doc_subsp_offset += 3;
			break;
		    
		    case 2:
			doc_line_table_output[output_offset++] 
				= DST_LN_DPC0_DLN1;
			doc_line_table_output[output_offset++] = line_num_delta;
			doc_subsp_offset += 2;
			break;

		    case 1:
			regular_entry.nibble.addr_delta = line_addr_delta; 
			regular_entry.nibble.line_delta = line_num_delta;
			doc_line_table_output[output_offset++] = 
			    regular_entry.byte;
			doc_subsp_offset++;
			break;
		}
	    } /* end ! need_first_statement */

	    break;

        case _DOC_ESC:

	    /*
	    ** Escape entries may have no following data.
	    */
	    if (escape_data_count > 0) {
	        doc_output(ESCAPE_DATA_SIZE);

                escape_data_count--;

	        /*
	        ** The data supplied as part of the R_STATEMENT fixup will
	        ** always fit within a byte when in escape mode.
	        */
	        doc_line_table_output[output_offset++] = line_num;
	        doc_subsp_offset++;
	    }

	    if (escape_data_count == 0)
		doc_state = _DOC_BUILD;
	    break;

	default:
	    doc_state = _DOC_DISABLED;
	    external_error(3011,0);
	    return;
	    break;
    }
}

/*
**************************************************************************
** 
** doc_finish()
**
** This routine causes any remaining buffered line table information to
** be written to the output file.  Since all the line tables have been
** built by this time the doc_info structure has the locations of all the
** line tables and the locations which should be patched with the locations
** of the line tables.  This routine patches the line table locations
** for every line table.  The patch location is determined by the symbol
** index and an offset which specified in the R_LINETAB fixup.
**
** Lastly, all allocated memory is free'd.
**
**************************************************************************
*/
void doc_finish(void)
{
    int index;
    int patch_loc;
    int sym_index;
    int subsp_sym_offset;
    int subsp_to_patch;

    /*
    ** Flush the output buffer if there is anything left.
    */
    doc_output(FLUSH_SIZE);

    /*
    ** patch specified locations with the subspace relative offset of
    ** the line table locations.
    */
    for (index = 0; index < num_line_tables; index++){
	if ((sym_index = doc_info[index].patch_symbol_index) != BAD_SYM) {
	    subsp_to_patch = Sym_Subsp(sym_index);

	    /*
	    ** determine the virtual offset relative to the start of the 
	    ** subspace.
	    */
            subsp_sym_offset = Subspace_Virtual_Offset(subsp_to_patch) -
			       symbol_value(doc_info[index].patch_symbol_index);

            /*
	    ** add the virtual offset with the actual file location of the
	    ** subspace to get actual file location of the symbol.  Lastly,
	    ** add in the offset off the symbol to determine patch location.
	    */
	    patch_loc = Subsp_Misc(subsp_to_patch).exec_file_offset +
			subsp_sym_offset +
			doc_info[index].patch_symbol_offset;
            
	    /*
	    ** Write the line table offset into the executable file at the
	    ** appropriate file location.
	    */
	    fdump(out_som_fd, 
		  patch_loc, 
		  (char *) &doc_info[index].line_table_offset, 
		  WORDSIZE, 
		  1);
	} else {
	    
	    /*
	    ** Internal error.  The symbol is either a bad symbol or was
	    ** incorrectly remapped.  Either way we cannot recover from this.
	    */
	    doc_state = _DOC_DISABLED;
	    external_error(3011,0);

	    /*
	    ** do not return until any allocated memory is made free.
	    */
	}
    }

    /*
    ** Clean up any garbage which is left over.
    */
    efree(doc_line_table_output);
    efree(doc_info);
}
