/*
 *  HP 9000 Series 800 Linker, Copyright Hewlett-Packard Co. 1985-1999  
 *  Symbol Strings/Space Strings Manipulation Routines
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  $Header: /home/cvs/cvsroot/linker/ld_strings.c,v 1.1.1.1 1999/10/18 19:53:02 pschwan Exp $
 */

#include <stdio.h>

#include "filehdr.h"

#include "std.h"
#include "driver.h"
#include "errors.h"
#include "ldlimits.h"
#include "linker.h"
#include "ld_strings.h"
#include "util.h"
#include "output.h"

/* Globals */

/* Buffers for holding raw strings for one .o as we read them from disk */
struct str_seg sym_str_buffer   = { NULL, 0 };
struct str_seg space_str_buffer = { NULL, 0 };

/* String table to hold strings from all input files */
struct string_table sym_strings   = { NULL, 0, NULL, 0,
				      0, NULL, 0, 0, 0,
				      NULL, NULL, 0 };

struct string_table tmp_strings   = { NULL, 0, NULL, 0,
				      0, NULL, 0, 0, 0,
				      NULL, NULL, 0 };

struct string_table space_strings = { NULL, 0, NULL, 0,
				      0, NULL, 0, 0, 0,
				      NULL, NULL, 0 };

/***************************************************************/
/*   String Hashing  *******************************************/
/***************************************************************/

unsigned int lib_hash_string(s)
register char *s;
{
    register unsigned int len, masklen;

    len = strlen(s);
    masklen = len & 0x7f;
    if (!len)
	internal_error(ZERO_LN_STRING , cur_name, 0);
    if (len == 1)
	return ((1 << 24) | (1 << 8) | (s[0] << 16) | s[0]);
    return ((masklen << 24) | (s[1] << 16) | (s[len-2] << 8) | s[len-1]);
} /* end lib_hash_string */

unsigned int hash_string(const char *s_arg)
{
    register unsigned int key;
    register unsigned char *s;

    s = (unsigned char *) s_arg;
    key = 0;
    while (*s)
	key = ((key << 5) | (key >> 27)) ^ *s++;
    return (key);
} /* end hash_string */

/***************************************************************/
/*   String Initialization  ************************************/
/***************************************************************/
				    
/******************************************************************************
* FUNCTION :            strings_initialize    
*
* ARGUMENTS:		
*	None
*       
* RETURN VALUE:
*       None
*
* PURPOSE:
*       Initialize string tables used by the linker. 
*
*       The 'sym_strings' string table is used for symbol strings
*       The 'tmp_strings' string table is used for temporary strings 
*            that will not be written to the output file. 
*       The 'space_strings' string table is used for space and subspace
*            names
*
******************************************************************************/

void strings_initialize()
{
    if (symbol_table_size < STRDUPHASHSIZE) {
       symbol_table_size = SYMBOL_STRING_HASH_SIZE;
    }
    gen_string_table(&sym_strings, 1024 * 64, TRUE, symbol_table_size);
    gen_string_table(&tmp_strings, MIN_STRBLK_ALLOCATION, FALSE, 0);
    gen_string_table(&space_strings, MIN_STRBLK_ALLOCATION, TRUE, STRDUPHASHSIZE);
}

/******************************************************************************
* FUNCTION :            gen_string_table    
*
* ARGUMENTS:		
*       hdl 	         - handle to string table to generate
*       allocation_hint  - hint to be used when allocating blocks of strings
*       build_hashtbl    - TRUE if we want a hash table to be built
*       
* RETURN VALUE:
*       None
*
* PURPOSE:
*      Delete any string table references by hdl if it exists and create	
*      a new one. 
*
* DATA STRUCTURE:
*      The symbol table consists of an array of segment records. Each
*      segment record points to a raw block of characters where the 
*      strings are kept and a size field that records the number of 
*      bytes in the block that have been used. We fill the segments
*      in sequential order, with fields in the handle structure 
*      recording where we are. 
*
*      An optional hash table consists of a linked list of uniform sized 
*      blocks of memory. The first word in each block is used as a next 
*      field to the next block. The hash table itself is placed in the 
*      first block in the list, right after the next field. Hash nodes are 
*      allocated out of the remaining blocks. 
* 
*      An optional array of segment records sorted by block_ptr address
*      can be created by calling the routine build_string_sort(). This 
*      array is used by the routine string_offset_sorted() to speed up 
*      the conversion of string pointer to string table offset during 
*      output. 
*
******************************************************************************/

void gen_string_table(hdl, allocation_hint, build_hashtbl, hash_bucket_size)
struct string_table *hdl;
int allocation_hint;
Boolean build_hashtbl;
int hash_bucket_size;
{
    char *node_blk;

    if (hdl->seg_array != NULL)
        purge_string_table(hdl);

    hdl->seg_array =
      (struct str_seg *) emalloc (sizeof (struct str_seg) * STRSEG_ALLOCATION);
    hdl->total_segments = STRSEG_ALLOCATION;
    hdl->open_segment = 0;  
    hdl->open_segment_bytes_free = allocation_hint;
    hdl->strblk_allocation_hint = allocation_hint; 
    hdl->seg_array[0].block_ptr = emalloc(allocation_hint);
    hdl->seg_array[0].size = 0;

    hdl->sort_array = NULL;
    hdl->cur_srch_segment = 0;

    if (build_hashtbl) {
        /*
           Allocate enough memory for the hash bucket pointers but
           at least STR_HASHNODE_ALLOCATION bytes to preserve the 
           allocation logic in add_string.
        */
        size_t node_blk_size = sizeof(int) + 
			       (sizeof(struct stringnode *) * 
			       hash_bucket_size);
        size_t node_blk_memsize = node_blk_size < STR_HASHNODE_ALLOCATION ?
                                  STR_HASHNODE_ALLOCATION : node_blk_size;
	node_blk = emalloc(node_blk_memsize);
	hdl->hash_table = (struct stringnode **) (node_blk + sizeof(int));
	hdl->free_hnode_offset = node_blk_size;
	hdl->free_hnode_blk = node_blk;
	hdl->hash_bucket_size = hash_bucket_size;
	memset(node_blk, '\0', hdl->free_hnode_offset);
    } else {
	hdl->hash_table = NULL;
	hdl->hash_bucket_size = 0;
	hdl->free_hnode_blk = NULL;
	hdl->free_hnode_offset = 0;
    }
} /* end gen_string_table */
    
/******************************************************************************
* FUNCTION :            purge_string_table_hash    
*
* ARGUMENTS:		
*	hdl - handle to string table
*       
* RETURN VALUE:
*       None
*
* PURPOSE:
*	Purge the hash table associated with the string table if the  
*       hash table exists. 
*
* DATA STRUCTURE
*       see gen_string_table()
*
******************************************************************************/

void purge_string_table_hash(hdl)
struct string_table *hdl;
{
    char *blk, *next_blk;

    if (hdl->hash_table != NULL) {
        blk = (((char *)hdl->hash_table) - sizeof(int));

	while (blk != NULL) {
	    next_blk = *(char **)blk;
	    efree(blk);
	    blk = next_blk;
        }

        hdl->hash_table = NULL;
    }
} /* end purge_string_table_hash */

/******************************************************************************
* FUNCTION :            purge_string_table    
*
* ARGUMENTS:		
*	hdl - handle to string table
*       
* RETURN VALUE:
*       None
*
* PURPOSE:
*       Purge the string table referenced by hdl	
*      
* DATA STRUCTURE:
*       see gen_string_table()
*
******************************************************************************/

void purge_string_table(hdl)
struct string_table *hdl;
{
    int i;

    for (i=0; i <= hdl->open_segment; i++) {
	if (hdl->seg_array[i].block_ptr != NULL) 
	    efree(hdl->seg_array[i].block_ptr);
    }

    efree(hdl->seg_array);
    hdl->seg_array = NULL;

    if (hdl->sort_array != NULL) {
        efree(hdl->sort_array);
	hdl->sort_array = NULL;
    }

    purge_string_table_hash(hdl);
} /* end purge_string_table */

/***************************************************************/
/*   Input Strings  ********************************************/
/***************************************************************/

/******************************************************************************
* FUNCTION :            strings_read    
*
* ARGUMENTS:		
*       fil_ofs    - file offset from which to read
*       size       - number of bytes to read
*       descriptor - string segment descriptor of where bytes will be placed
*	hdl        - handle of string table where strings will be eventually
*                    placed by add_string().
*       
* RETURN VALUE:
*       None
*
* PURPOSE:
*       Read the strings into the buffer indicated by descriptor. 	
*
******************************************************************************/

void strings_read(fil_ofs, size, descriptor, hdl)
int fil_ofs;
int size;
struct str_seg *descriptor;
struct string_table *hdl;
{

    /* Make sure the buffer is big enough. Since this is a buffer and    */
    /* we really do not want to preserve it's contents, it makes more    */
    /* sense to free the region and then do a malloc, rather than a      */
    /* realloc								 */
    if (descriptor->size < size) {
        if (descriptor->size > 0) 
	    efree(descriptor->block_ptr);
	descriptor->size = max(MIN_STRREAD_ALLOCATION, size + (size / 4));
	descriptor->block_ptr = emalloc(descriptor->size);
    }

    /* read the strings into the new region */
    ffetch(cur_fildes, fil_ofs, descriptor->block_ptr, 1, size);

    /* 
    ** set allocation hint for transferring these strings to the string table
    */
    hdl->strblk_allocation_hint = max(MIN_STRBLK_ALLOCATION, 
				      descriptor->size);
}

/******************************************************************************
* FUNCTION :            add_string    
*
* ARGUMENTS:		
*       hdl           - handle to string table 
*       str_ptr       - pointer to string to be added
*       hashval       - hash value for string
*       length_prefix - TRUE if input string is preceeded by length word
*       use_hash      - TRUE if we should use hash table to lookup/store string
*                         Certain strings have a low probabilty of having a 
*                         duplicate, so it does not pay to look for them in 
*                         the hash table or store them there. 
*       
* RETURN VALUE:
*       pointer to string in the string table
*
* PURPOSE:
*       add the given string to the string table referenced by hdl.	
*
* DATA STRUCTURE:
*       see gen_string_table(). 
*
*       All strings entered into the string table are prefixed with  
*       a word containing the length of the string. Because of this,
*       strings must be aligned on word boundaries   
*
******************************************************************************/

char *add_string(hdl, str_ptr, hashval, length_prefix, use_hash)
struct string_table *hdl;
char *str_ptr;
unsigned int hashval;
Boolean length_prefix;
Boolean use_hash;
{
    int str_len, bytes_needed;
    struct stringnode *n;
    struct str_seg *seg_ptr;
    char *dest, *new_blk;
    unsigned int hashkey;
    struct stringnode **hash_table;

    hash_table = hdl->hash_table;

    if (length_prefix) 
        str_len = *(((int *)str_ptr) -1);
    else 
        str_len = strlen(str_ptr);  

    /* if hash table is present, search for a duplicate entry */
    if (hash_table != NULL && use_hash) {

        hashkey = hashval % hdl->hash_bucket_size;
        n = hash_table[hashkey];
    
        while (n != NULL) {
	    if (n->hashval == hashval && 
                *(((int *)n->str_ptr) -1) == str_len &&
                strcmp(n->str_ptr, str_ptr) == 0) {
		/* found a duplicate */
	        return(n->str_ptr);
		}
	    n = n->next;
	    }
	}

    /* add string to table */
   
    /* add in room for length field, null byte, and round to word length */
    bytes_needed = sizeof(int) + ((str_len + sizeof(int)) & ~(sizeof(int)-1));

    if (bytes_needed > hdl->open_segment_bytes_free) {
        if (++hdl->open_segment >= hdl->total_segments) {
            hdl->total_segments += STRSEG_ALLOCATION;
	    hdl->seg_array = (struct str_seg *) 
	                       erealloc ((char *) hdl->seg_array, 
					 hdl->total_segments 
					     * sizeof(struct str_seg));
        }

        hdl->open_segment_bytes_free = hdl->strblk_allocation_hint;
        hdl->seg_array[hdl->open_segment].block_ptr = 
                               emalloc(hdl->open_segment_bytes_free);
        hdl->seg_array[hdl->open_segment].size = 0;
    }

    seg_ptr = &hdl->seg_array[hdl->open_segment];
    dest = seg_ptr->block_ptr + seg_ptr->size;

    /* zero out the trailing pad bytes. /bin/ar assumes that the  */
    /* last byte in the string table will be '\0' 	          */
    *((int *)(dest + bytes_needed - sizeof(int))) = 0;

    /* copy the length over */
    *(int *)dest = str_len;
    dest += sizeof(int);

    strcpy(dest,str_ptr);

    seg_ptr->size += bytes_needed;
    hdl->open_segment_bytes_free -= bytes_needed;

    /* if a hash table is present, add string to hash table */
    if (hash_table != NULL && use_hash) {

	if (hdl->free_hnode_offset + sizeof(struct stringnode) > 
						 STR_HASHNODE_ALLOCATION) {
	    new_blk = emalloc(STR_HASHNODE_ALLOCATION);
	    hdl->free_hnode_offset = sizeof(int);
	    *((int *)new_blk) = *(((int *)hash_table) -1);
	    *(((int *)hash_table) -1) = (int)new_blk;
	    hdl->free_hnode_blk = new_blk;
	}

	n = (struct stringnode *)(hdl->free_hnode_blk + 
				  hdl->free_hnode_offset);
	hdl->free_hnode_offset += sizeof(struct stringnode);

	n->str_ptr = dest;
	n->hashval = hashval;
	n->next = hash_table[hashkey];
        hash_table[hashkey] = n;
    }
    return(dest);
} /* end add_string */

/***************************************************************/
/*   Output Strings  *******************************************/
/***************************************************************/

/******************************************************************************
* FUNCTION :            out_string_table    
*
* ARGUMENTS:		
*	hdl        - handle to string table
*	header_loc - location of where bytes are to be written  
*       
* RETURN VALUE:
*       Total bytes written
*
* PURPOSE:
*       write the string bytes from the given string table to the output
*       file.
*
* DATA STRUCTURE:
*       see gen_string_table(). 
*
******************************************************************************/

int out_string_table(hdl, header_loc)
struct string_table *hdl;
int    header_loc;
{
    int i, len, total;

    total = 0;
    for (i=0; i <= hdl->open_segment; i++) {
        len = hdl->seg_array[i].size;
        fdump (out_som_fd, header_loc, hdl->seg_array[i].block_ptr, 1, len);
        header_loc += len;
        total += len;
    }

    return(total);
} 

/***************************************************************/
/*   Query Strings  ********************************************/
/***************************************************************/

/******************************************************************************
* FUNCTION :            sizeof_string_table    
*
* ARGUMENTS:		
*	hdl - handle to string table
*       
* RETURN VALUE:
*       total number of bytes in the string table  
*
* PURPOSE:
*	return the total number of bytes in the string table	
*
* DATA STRUCTURE:
*       see gen_string_table(). 
*
******************************************************************************/

int sizeof_string_table(hdl)
struct string_table *hdl;
{
    int i;
    int total;

    total = 0;
    for (i=0; i <= hdl->open_segment; i++) 
        total += hdl->seg_array[i].size;

    return(total);
}

/******************************************************************************
* FUNCTION :            string_ptr_linear    
*
* ARGUMENTS:		
*	offset - offset into the string table
*	hdl    - handle of the string table
*       
* RETURN VALUE:
*       character pointer for the string located at offset.
*
* PURPOSE:
*       convert an offset into the string table into the actual string.	
*
* DATA STRUCTURE:
*       see gen_string_table(). 
*
******************************************************************************/

char *string_ptr_linear(offset, hdl)
int offset;
struct string_table *hdl;
{
    int i;
    int blk_offset;

    blk_offset = 0;
    for (i=0; i <= hdl->open_segment; i++) {
        blk_offset += hdl->seg_array[i].size;	
        if (offset < blk_offset) {
	    return (hdl->seg_array[i].block_ptr + offset);
       	}
    } /* for */
    internal_error(BAD_STRING, 0);
    return NULL;
}

/******************************************************************************
* FUNCTION :            string_offset_linear    
*
* ARGUMENTS:		
*	s   - pointer to string located in string table
*	hdl - handle to string table
*       
* RETURN VALUE:
*       offset into string table
*
* PURPOSE:
*       convert the given character pointer into a byte offset into	
*       the string table. 
*
* ALGORITHM:
*       The segments in the string table are searched in linear order
*       looking for the block that contains the pointer. The routine
*       string_offset_sorted() should be used if a large number of 
*       segments are expected. 
*
* DATA STRUCTURE:
*       see gen_string_table(). 
*
******************************************************************************/

int string_offset_linear(s, hdl)
register char *s;
struct string_table *hdl;
{
    int i;
    int offset, block_size;
    char *block_ptr;

    offset = 0;
    for (i=0; i <= hdl->open_segment; i++) {
	block_ptr = hdl->seg_array[i].block_ptr;
	block_size = hdl->seg_array[i].size; 
        if (s >= block_ptr && s < block_ptr + block_size)
	    return (offset + (s - block_ptr));
        offset += block_size;
    }

    internal_error(BAD_STRING, 0);
    return 0;
}

/******************************************************************************
* FUNCTION :            compare_str_seg    
*
* ARGUMENTS:		
*	ptr1 - pointer to record being sorted
*	ptr2 - pointer to record being sorted
*       
* RETURN VALUE:
*       <0 if ptr1 < ptr2, 0 if ptr1 == ptr2, >0 if ptr1 > ptr2  
*
* PURPOSE:
*       used by qsort to sort segment blocks by ascending address
*
******************************************************************************/

compare_str_seg(ptr1, ptr2)
struct str_seg *ptr1, *ptr2;
{
    return(ptr1->block_ptr - ptr2->block_ptr);
}

/******************************************************************************
* FUNCTION :            build_string_sort    
*
* ARGUMENTS:		
*	hdl - handle to string table
*       
* RETURN VALUE:
*       None
*
* PURPOSE:
*       build an array of sorted segment records used by the routine
*       string_offset_sorted().  
*
* DATA STRUCTURE:
*       see gen_string_table(). 
*
*       The sort_array is an image of the seg_array that has been sorted
*       by block_ptr address. The size field is used to reference back to
*       the original index of the segment. 
*
*       The inorder array is an image of the seg_array. The size field 
*       is the byte offset of the beginning of this segment.
*
******************************************************************************/

void build_string_sort(hdl)
struct string_table *hdl;
{
    int i, block_offset;

    /* allocate sort array, +1 since open_segment is an array index starting */
    /* at zero, +1 for a dummy record at the end used in searching           */
    hdl->sort_array =
	  (struct str_seg *) emalloc (((hdl->open_segment +2) * 2) * 
			                          sizeof(struct str_seg)); 
    hdl->inorder_array = &hdl->sort_array[hdl->open_segment +2];
    hdl->cur_srch_segment = 0;

    block_offset = 0;
    for (i = 0; i <= hdl->open_segment; i++) {
	hdl->sort_array[i].block_ptr     = 
	hdl->inorder_array[i].block_ptr  = hdl->seg_array[i].block_ptr;
	hdl->sort_array[i].size       = i; 
	hdl->inorder_array[i].size    = block_offset; 
	block_offset += hdl->seg_array[i].size; 
	}
    /* dummy block at the end of the array used to bounds check the last elt */
    hdl->sort_array[i].block_ptr = (char *)MAXINT;
    hdl->sort_array[i].size      = i; 

    qsort(hdl->sort_array, hdl->open_segment + 1, 
	  sizeof(struct str_seg), compare_str_seg);
}

/******************************************************************************
* FUNCTION :            str_seg_search    
*
* ARGUMENTS:		
*       low     - low bounds of search
*       high    - high bounds of search
*       seg_ptr - segment array being searched
*       s       - character pointer we are trying to find
*       
* RETURN VALUE:
*       index of the segment in which the string was found, else 
*       return -1.
*
* PURPOSE:
*       search the string segments between 'low' and 'high' inclusive for 	
*       a string segment containing 's'. 
*
* ALGORITHM:
*       Binary search. 
*
* DATA STRUCTURE:
*       see gen_string_table(). 
*
*       The last record in the segment array is a dummy entry that 
*       is used in bounds checking. 
*
******************************************************************************/

static int str_seg_search(low, high, seg_ptr, s)
int low;
int high;
struct str_seg *seg_ptr;
char *s;
{
    int middle;

    while (low <= high) {
        middle = (low + high) / 2;
        if (s >= seg_ptr[middle].block_ptr && 
	    s < seg_ptr[middle+1].block_ptr) {
            return(seg_ptr[middle].size);
        } else if (s > seg_ptr[middle].block_ptr) {
            low = middle +1;
        } else {
            high = middle -1;
        }
    }

    return(-1);
}

/******************************************************************************
* FUNCTION :            string_offset_sorted    
*
* ARGUMENTS:		
*	s   - pointer to character string present in string table
*	hdl - handle of string table
*       
* RETURN VALUE:
*       offset in to string table for string referenced by s
*
* PURPOSE:
*       convert the character pointer given to a byte offset into the 
*       string table 
*
* ALGORITHM:
*       Assume that requests for conversion are coming in the same 
*       order in which the strings were added to the table. 
*
*       Assume that build_string_sort() has been called before this routine
*       for the given string table. 
*
*       Keep a record of the last segment in which we found a match and
*       look in that segment and the next one for the pointer. 
*       If we did not find the string, use a binary search to look 
*       for the ptr. 
*
* DATA STRUCTURE:
*       see gen_string_table(). 
*
******************************************************************************/

int string_offset_sorted(s, hdl)
register char *s;
struct string_table *hdl;
{
    int i, loops;
    struct str_seg *seg_ptr;

    seg_ptr = &hdl->seg_array[hdl->cur_srch_segment];

    /* look in current string block being searched and the next one */
    loops = hdl->cur_srch_segment == hdl->open_segment ? 0 : 1;
    i = 0;
    do {
        if (s >= seg_ptr->block_ptr && 
	    s <  seg_ptr->block_ptr + seg_ptr->size) {
	    hdl->cur_srch_segment += i;
            seg_ptr = &hdl->inorder_array[hdl->cur_srch_segment];
	    return((s - seg_ptr->block_ptr) + seg_ptr->size);
	    }
	++seg_ptr;
    } while (i++ < loops);

    /* didn't find it in current block or the next one, */
    /* so resort to binary search			*/                

    i = str_seg_search(0, hdl->open_segment, hdl->sort_array, s);

    if (i != -1) {
	hdl->cur_srch_segment = i;
        seg_ptr = &hdl->inorder_array[i];
	return((s - seg_ptr->block_ptr) + seg_ptr->size);
    }

    internal_error(BAD_STRING, 0);
    return 0;
}
